<?php
require_once __DIR__ . '/../includes/auth.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/functions.php';
require_login();

// Restrict to Cashier or Caissier_Comptable role
if (!in_array($_SESSION['role_name'], ['Cashier', 'Caissier_Comptable'])) {
    error_log("Unauthorized access to cashier_dashboard.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
    header("Location: /masunzu_bar_hotel/dashboards/operation_dashboard.php");
    exit;
}

// Fetch province_id if not set
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in cashier_dashboard.php");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../includes/footer.php';
    exit;
}
if (!isset($_SESSION['province_id']) || $_SESSION['province_id'] === '') {
    $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
    if (!$stmt) {
        error_log("Query preparation failed for province_id: " . $conn->error);
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur de base de données. Contactez l'administrateur.</p>";
        include __DIR__ . '/../includes/footer.php';
        exit;
    }
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $_SESSION['province_id'] = $user['province_id'] ?? null;
    $stmt->close();
}
$province_id = $_SESSION['province_id'] ?? null;

if ($province_id === null || $province_id === '') {
    error_log("No province_id for user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur : Aucun dépôt assigné. Contactez l'administrateur.</p>";
    include __DIR__ . '/../includes/footer.php';
    exit;
}
$province_id = (int)$province_id;

$notificationMessages = [];
$createNotificationSql = "
    CREATE TABLE IF NOT EXISTS user_notification (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        message VARCHAR(255) NOT NULL,
        link VARCHAR(255) DEFAULT NULL,
        is_read TINYINT(1) NOT NULL DEFAULT 0,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_read (user_id, is_read),
        CONSTRAINT fk_notification_user_dashboard FOREIGN KEY (user_id) REFERENCES user(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";
if (!$conn->query($createNotificationSql)) {
    error_log('Failed to ensure notification table on cashier dashboard: ' . $conn->error);
}

$cashierNotifications = [];
$stmtNotif = $conn->prepare("SELECT id, message, link, created_at FROM user_notification WHERE user_id = ? AND is_read = 0 ORDER BY created_at DESC LIMIT 5");
if ($stmtNotif) {
    $stmtNotif->bind_param("i", $_SESSION['user_id']);
    $stmtNotif->execute();
    $cashierNotifications = $stmtNotif->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmtNotif->close();

    if (!empty($cashierNotifications)) {
        $ids = array_map('intval', array_column($cashierNotifications, 'id'));
        $idList = implode(',', $ids);
        $stmtMark = $conn->prepare("UPDATE user_notification SET is_read = 1 WHERE user_id = ? AND id IN ($idList)");
        if ($stmtMark) {
            $stmtMark->bind_param('i', $_SESSION['user_id']);
            $stmtMark->execute();
            $stmtMark->close();
        }
    }
} else {
    error_log('Failed to fetch notifications for cashier: ' . $conn->error);
}

// Fetch dépôt name
$stmt = $conn->prepare("SELECT name FROM province WHERE id = ?");
if (!$stmt) {
    error_log("Query preparation failed for province name: " . $conn->error);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database error. Contact administrator.</p>";
    include __DIR__ . '/../includes/footer.php';
    exit;
}
$stmt->bind_param("i", $province_id);
$stmt->execute();
$province = $stmt->get_result()->fetch_assoc()['name'] ?? 'Unknown';
$stmt->close();

// Check active shift
$active_shift = null;
$stmt = $conn->prepare("SELECT id, shift_start, beginning_balance, deposits, end_balance FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL ORDER BY shift_start DESC LIMIT 1");
if (!$stmt) {
    error_log("Query preparation failed for active shift: " . $conn->error);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database error. Contact administrator.</p>";
    include __DIR__ . '/../includes/footer.php';
    exit;
}
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$result = $stmt->get_result();
$active_shift = $result->fetch_assoc() ?: null;
$stmt->close();

// Calculate current balance for active shift (same calculation as pay_requisitions.php)
$shift_deposits = 0;
$shift_expenses = 0;
$shift_transfers = 0;
$current_balance = 0;
$shift_start = $active_shift['shift_start'] ?? null;

if ($active_shift && $shift_start) {
    $beginning_balance = (float)($active_shift['beginning_balance'] ?? 0);
    $shift_deposits = (float)($active_shift['deposits'] ?? 0);

    // Calculate expenses for this shift (from shift_start to now)
    $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_expenses FROM expense WHERE cashier_id = ? AND created_at >= ?");
    if (!$stmt) {
        error_log("Query preparation failed for shift expenses: " . $conn->error);
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur de base de données. Contactez l'administrateur.</p>";
        include __DIR__ . '/../includes/footer.php';
        exit;
    }
    $stmt->bind_param("is", $_SESSION['user_id'], $shift_start);
    $stmt->execute();
    $shift_expenses = (float)($stmt->get_result()->fetch_assoc()['total_expenses'] ?? 0);
    $stmt->close();

    // Calculate transfers for this shift (from shift_start to now)
    $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_transfers FROM cash_transfer WHERE cashier_id = ? AND transfer_date >= ?");
    if (!$stmt) {
        error_log("Query preparation failed for shift transfers: " . $conn->error);
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur de base de données. Contactez l'administrateur.</p>";
        include __DIR__ . '/../includes/footer.php';
        exit;
    }
    $stmt->bind_param("is", $_SESSION['user_id'], $shift_start);
    $stmt->execute();
    $shift_transfers = (float)($stmt->get_result()->fetch_assoc()['total_transfers'] ?? 0);
    $stmt->close();
    
    // Calculate current balance: beginning_balance + deposits - expenses - transfers
    $current_balance = $beginning_balance + $shift_deposits - $shift_expenses - $shift_transfers;
    if ($current_balance < 0) {
        $current_balance = 0;
    }
}

// Ensure cashier_account table exists
$createCashierAccountSql = "
    CREATE TABLE IF NOT EXISTS cashier_account (
        id INT AUTO_INCREMENT PRIMARY KEY,
        cashier_id INT NOT NULL,
        shift_id INT NULL,
        transaction_type ENUM('SHIFT_START','SHIFT_END','INVOICE_PAYMENT','LOAN_REPAYMENT','REQUISITION_PAYMENT','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','ADJUSTMENT') NOT NULL,
        direction ENUM('CREDIT','DEBIT') NOT NULL,
        amount BIGINT(20) NOT NULL,
        reference_type ENUM('SHIFT','INVOICE','LOAN_PAYMENT','REQUISITION','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','MANUAL') NOT NULL,
        reference_id INT NULL,
        description VARCHAR(255) DEFAULT NULL,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        created_by INT NULL,
        INDEX idx_cashier_created (cashier_id, created_at),
        INDEX idx_shift (shift_id),
        INDEX idx_reference (reference_type, reference_id),
        CONSTRAINT fk_ca_cashier FOREIGN KEY (cashier_id) REFERENCES user(id) ON DELETE CASCADE,
        CONSTRAINT fk_ca_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE SET NULL,
        CONSTRAINT fk_ca_creator FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE SET NULL
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";
@$conn->query($createCashierAccountSql);

// CALCULATE ACTUAL CASH IN ACCOUNT
// Logic: Money from closed shifts (not deposited/transferred) + Current cash in active shift
$actual_cash_in_account = 0.00;

// Step 1: Calculate available money from closed shifts
// For each closed shift, get end_balance minus any deposits/transfers linked to that shift
$available_from_closed_shifts = 0.00;
$cumulative_from_closed_shifts = 0.00;

// Check if new YES/NO tracking columns exist in cashier_shift_closure
$has_yesno_columns = false;
$checkColumns = $conn->query("
    SELECT COLUMN_NAME 
    FROM INFORMATION_SCHEMA.COLUMNS 
    WHERE TABLE_SCHEMA = DATABASE() 
    AND TABLE_NAME = 'cashier_shift_closure' 
    AND COLUMN_NAME = 'deposited_to_bank'
");
if ($checkColumns && $checkColumns->num_rows > 0) {
    $has_yesno_columns = true;
    $checkColumns->close();
}

// Get all closed shifts with their end_balance and YES/NO tracking info
if ($has_yesno_columns) {
    // New method: Use YES/NO flags from cashier_shift_closure
    $stmt = $conn->prepare("
        SELECT 
            cb.id,
            cb.shift_start,
            cb.shift_end,
            cb.beginning_balance,
            cb.end_balance,
            cb.deposits,
            cb.withdrawals,
            COALESCE(csc.deposited_to_bank, 'no') AS deposited_to_bank,
            COALESCE(csc.transferred_to_accountant, 'no') AS transferred_to_accountant,
            csc.counted_amount
        FROM cashier_balance cb
        LEFT JOIN cashier_shift_closure csc ON cb.id = csc.shift_id
        WHERE cb.cashier_id = ? AND cb.shift_end IS NOT NULL
        ORDER BY cb.shift_end DESC
    ");
} else {
    // Old method: Get basic shift info, will check junction tables later
    $stmt = $conn->prepare("
        SELECT 
            cb.id,
            cb.shift_start,
            cb.shift_end,
            cb.beginning_balance,
            cb.end_balance,
            cb.deposits,
            cb.withdrawals
        FROM cashier_balance cb
        WHERE cb.cashier_id = ? AND cb.shift_end IS NOT NULL
        ORDER BY cb.shift_end DESC
    ");
}
if ($stmt) {
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $closed_shifts = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
    
    // For each closed shift, calculate available amount
    foreach ($closed_shifts as $shift) {
        $shift_id = (int)$shift['id'];
        // Use end_balance if available, otherwise calculate from beginning_balance + deposits - withdrawals
        $end_balance = null;
        if (isset($shift['end_balance']) && $shift['end_balance'] !== null) {
            $end_balance = (float)$shift['end_balance'];
        } else {
            // Calculate if end_balance is NULL
            $beginning = (float)($shift['beginning_balance'] ?? 0);
            $deposits = (float)($shift['deposits'] ?? 0);
            $withdrawals = (float)($shift['withdrawals'] ?? 0);
            $end_balance = $beginning + $deposits - $withdrawals;
        }
        
        // Check if shift has been deposited or transferred using YES/NO flags
        if ($has_yesno_columns) {
            // New method: Check YES/NO flags
            // If cashier_shift_closure record doesn't exist, default to 'no' (money is still available)
            $deposited_flag = $shift['deposited_to_bank'] ?? 'no';
            $transferred_flag = $shift['transferred_to_accountant'] ?? 'no';
            $is_deposited = strtolower($deposited_flag) === 'yes';
            $is_transferred = strtolower($transferred_flag) === 'yes';
        } else {
            // Old method: Check junction tables (backward compatibility)
            $is_deposited = false;
            $is_transferred = false;
            
            $stmtDeposit = $conn->prepare("
                SELECT COUNT(*) AS has_deposit
                FROM cashier_shift_deposit csd
                INNER JOIN bank_deposit bd ON csd.deposit_id = bd.id
                WHERE csd.shift_id = ?
            ");
            if ($stmtDeposit) {
                $stmtDeposit->bind_param("i", $shift_id);
                $stmtDeposit->execute();
                $depositResult = $stmtDeposit->get_result()->fetch_assoc();
                $is_deposited = (int)($depositResult['has_deposit'] ?? 0) > 0;
                $stmtDeposit->close();
            }
            
            $stmtTransfer = $conn->prepare("
                SELECT COUNT(*) AS has_transfer
                FROM cashier_shift_transfer cst
                INNER JOIN cash_transfer ct ON cst.transfer_id = ct.id
                WHERE cst.shift_id = ? AND ct.status = 'received'
            ");
            if ($stmtTransfer) {
                $stmtTransfer->bind_param("i", $shift_id);
                $stmtTransfer->execute();
                $transferResult = $stmtTransfer->get_result()->fetch_assoc();
                $is_transferred = (int)($transferResult['has_transfer'] ?? 0) > 0;
                $stmtTransfer->close();
            }
        }
        
        // Available from this shift = counted_amount (actual cash) if available, otherwise end_balance
        // If either deposited_to_bank='yes' OR transferred_to_accountant='yes', money is no longer available
        $available_amount = $end_balance;
        
        // Use counted_amount from cashier_shift_closure if available (this is the actual cash counted)
        if ($has_yesno_columns && isset($shift['counted_amount']) && $shift['counted_amount'] !== null) {
            $available_amount = (float)$shift['counted_amount'];
        }
        
        // Only add if amount > 0 and not deposited/transferred
        if ($available_amount > 0 && !$is_deposited && !$is_transferred) {
            $available_from_closed_shifts += $available_amount;
        }
        
        // For display: cumulative end_balance (before subtracting deposits/transfers)
        // Show actual sum including negative values for transparency
        $cumulative_from_closed_shifts += $end_balance;
    }
}

// Step 2: Get total bank deposits for display (all deposits, not just from closed shifts)
$total_bank_deposits = 0.00;
$stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_deposits FROM bank_deposit WHERE user_id = ?");
if ($stmt) {
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $total_bank_deposits = (float)($stmt->get_result()->fetch_assoc()['total_deposits'] ?? 0);
    $stmt->close();
}

// Step 3: Get total transfers to accountant for display (all transfers, not just from closed shifts)
$total_transfers_to_accountant = 0.00;
$stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_transferred FROM cash_transfer WHERE cashier_id = ? AND status = 'received'");
if ($stmt) {
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $total_transfers_to_accountant = (float)($stmt->get_result()->fetch_assoc()['total_transferred'] ?? 0);
    $stmt->close();
}

// Step 4: Calculate Argent Disponible (closed shifts money + active shift money)
// Argent Disponible = Money from closed shifts (not deposited/transferred) + Active shift money
$argent_disponible = $available_from_closed_shifts;
if ($active_shift && $shift_start) {
    $argent_disponible += $current_balance;
}
if ($argent_disponible < 0) {
    $argent_disponible = 0;
}

// Sessions fermées = amount from closed shifts NOT sent to accountant or bank
// This is already calculated as $available_from_closed_shifts

// Calculate bank deposits for active shift (for display in session status)
$active_shift_bank_deposits = 0.00;
if ($active_shift && isset($active_shift['id'])) {
    $active_shift_id = (int)$active_shift['id'];
    $stmtBankDeposits = $conn->prepare("
        SELECT COALESCE(SUM(bd.amount), 0) AS total_deposits
        FROM bank_deposit bd
        INNER JOIN cashier_shift_deposit csd ON bd.id = csd.deposit_id
        WHERE csd.shift_id = ?
    ");
    if ($stmtBankDeposits) {
        $stmtBankDeposits->bind_param("i", $active_shift_id);
        $stmtBankDeposits->execute();
        $bankDepositResult = $stmtBankDeposits->get_result()->fetch_assoc();
        $active_shift_bank_deposits = (float)($bankDepositResult['total_deposits'] ?? 0);
        $stmtBankDeposits->close();
    }
}

// Calculate accumulated cash from current active shift
$accumulated_cash_this_shift = 0.00;
if ($active_shift && isset($active_shift['id'])) {
    $shift_id = (int)$active_shift['id'];
    $stmt = $conn->prepare("
        SELECT 
            COALESCE(SUM(CASE WHEN direction = 'CREDIT' AND transaction_type != 'SHIFT_START' THEN amount ELSE 0 END), 0) AS shift_credits,
            COALESCE(SUM(CASE WHEN direction = 'DEBIT' THEN amount ELSE 0 END), 0) AS shift_debits
        FROM cashier_account
        WHERE cashier_id = ? AND shift_id = ?
    ");
    if ($stmt) {
        $stmt->bind_param("ii", $_SESSION['user_id'], $shift_id);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        $shift_credits = (float)($result['shift_credits'] ?? 0);
        $shift_debits = (float)($result['shift_debits'] ?? 0);
        $stmt->close();
        $accumulated_cash_this_shift = $shift_credits - $shift_debits;
        if ($accumulated_cash_this_shift < 0) {
            $accumulated_cash_this_shift = 0;
        }
    }
}

// Calculate pending transfers for display (still needed for UI)
$total_transfers_pending = 0.00;
$stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_pending FROM cash_transfer WHERE cashier_id = ? AND status = 'pending'");
if (!$stmt) {
    error_log("Query preparation failed for pending transfers: " . $conn->error);
} else {
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $total_transfers_pending = (float)($stmt->get_result()->fetch_assoc()['total_pending'] ?? 0);
    $stmt->close();
}

// Variables already calculated above for actual_cash_in_account calculation
// $total_bank_deposits, $total_transfers_to_accountant, and $cumulative_from_closed_shifts are already set
// Use $total_transfers_to_accountant as $total_transfers_received for display consistency
$total_transfers_received = $total_transfers_to_accountant;

// Fetch metrics
$today_start = date('Y-m-d 00:00:00');
$stmt = $conn->prepare("SELECT SUM(paid_amount) as total_sales FROM `order` WHERE created_by = ? AND province_id = ? AND paid_status = 'Paid' AND created_at >= ?");
if (!$stmt) {
    error_log("Query preparation failed for total sales (orders): " . $conn->error);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database error. Contact administrator.</p>";
    include __DIR__ . '/../includes/footer.php';
    exit;
}
$stmt->bind_param("iis", $_SESSION['user_id'], $province_id, $today_start);
$stmt->execute();
$total_sales_orders = (int)($stmt->get_result()->fetch_assoc()['total_sales'] ?? 0);
$stmt->close();

$stmt = $conn->prepare("SELECT SUM(paid_amount) as total_sales FROM paid_invoices WHERE created_by = ? AND province_id = ? AND status = 'paid' AND paid_at >= ?");
if (!$stmt) {
    error_log("Query preparation failed for total sales (invoices): " . $conn->error);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database error. Contact administrator.</p>";
    include __DIR__ . '/../includes/footer.php';
    exit;
}
$stmt->bind_param("iis", $_SESSION['user_id'], $province_id, $today_start);
$stmt->execute();
$total_sales_invoices = (int)($stmt->get_result()->fetch_assoc()['total_sales'] ?? 0);
$stmt->close();

$total_sales = $total_sales_orders + $total_sales_invoices;

// Fetch loan repayments for today
$credit_sales_today = 0;
$credit_sales_count = 0;
$stmtLoan = $conn->prepare("
    SELECT COALESCE(SUM(cll.amount_bif), 0) AS total_amount,
           COUNT(*) AS sale_count
    FROM customer_loan_ledger cll
    LEFT JOIN unpaid_invoices ui ON cll.reference_table = 'unpaid_invoices' AND cll.reference_id = ui.id
    WHERE cll.entry_type = 'loan_sale'
      AND cll.direction = 'debit'
      AND cll.created_by = ?
      AND DATE(cll.created_at) = CURDATE()
      AND (ui.province_id = ? OR ui.province_id IS NULL)
");
if ($stmtLoan) {
    $stmtLoan->bind_param("ii", $_SESSION['user_id'], $province_id);
    $stmtLoan->execute();
    $loanResult = $stmtLoan->get_result()->fetch_assoc();
    $credit_sales_today = (int)($loanResult['total_amount'] ?? 0);
    $credit_sales_count = (int)($loanResult['sale_count'] ?? 0);
    $stmtLoan->close();
} else {
    error_log('Failed to prepare loan sales query: ' . $conn->error);
}

$stmt = $conn->prepare("SELECT COUNT(*) as pending_orders FROM `order` WHERE province_id = ? AND serve_status = 'Pending'");
if (!$stmt) {
    error_log("Query preparation failed for pending orders: " . $conn->error);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database error. Contact administrator.</p>";
    include __DIR__ . '/../includes/footer.php';
    exit;
}
$stmt->bind_param("i", $province_id);
$stmt->execute();
$pending_orders = $stmt->get_result()->fetch_assoc()['pending_orders'] ?? 0;
$stmt->close();



$province_invoice_summary = [
    'pending' => 0,
    'approved' => 0,
    'rejected' => 0,
    'paid' => 0,
    'served' => 0,
];

$provinceStatusStmt = $conn->prepare("
    SELECT approval_status, COUNT(*) AS total
    FROM unpaid_invoices
    WHERE province_id = ?
    GROUP BY approval_status
");
if ($provinceStatusStmt) {
    $provinceStatusStmt->bind_param("i", $province_id);
    $provinceStatusStmt->execute();
    $statusResult = $provinceStatusStmt->get_result();
    while ($row = $statusResult->fetch_assoc()) {
        $key = strtolower($row['approval_status']);
        if (isset($province_invoice_summary[$key])) {
            $province_invoice_summary[$key] = (int)$row['total'];
        }
    }
    $provinceStatusStmt->close();
} else {
    error_log('Failed to prepare province invoice status summary: ' . $conn->error);
}

$provincePaidStmt = $conn->prepare("
    SELECT 
        COUNT(*) AS paid_total,
        SUM(CASE WHEN serve_status = 'Served' THEN 1 ELSE 0 END) AS served_total
    FROM paid_invoices
    WHERE province_id = ?
");
if ($provincePaidStmt) {
    $provincePaidStmt->bind_param("i", $province_id);
    $provincePaidStmt->execute();
    $paidRow = $provincePaidStmt->get_result()->fetch_assoc();
    $province_invoice_summary['paid'] = (int)($paidRow['paid_total'] ?? 0);
    $province_invoice_summary['served'] = (int)($paidRow['served_total'] ?? 0);
    $provincePaidStmt->close();
} else {
    error_log('Failed to prepare province paid invoice summary: ' . $conn->error);
}

$conn->close();

require_once __DIR__ . '/../includes/header.php';
?>
<div class="dashboard-container" style="padding: 20px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 100vh; width: 100%; box-sizing: border-box;">
    <div class="dashboard-content" style="background-color: #FFFFFF; padding: 30px; border-radius: 12px; box-shadow: 0 4px 20px rgba(0,0,0,0.15); max-width: 100%; width: 100%; margin: 0; box-sizing: border-box;">
        <div style="text-align: center; margin-bottom: 30px; padding-bottom: 20px; border-bottom: 2px solid #E0E0E0;">
            <h2 style="color: #4B2F1F; margin-bottom: 8px; font-weight: 600; font-size: 32px;">Tableau de Bord Caissier</h2>
            <p style="color: #666; font-size: 16px; margin: 0;">Bienvenue, <?php echo htmlspecialchars($_SESSION['full_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8'); ?> (Caissier, Dépôt : <?php echo htmlspecialchars($province ?? 'Inconnu', ENT_QUOTES, 'UTF-8'); ?>)</p>
        </div>

        <?php if (!empty($cashierNotifications)): ?>
            <div class="alert alert-success" style="border-radius: 10px;">
                <div style="font-weight: 600; margin-bottom: 8px; display: flex; align-items: center; gap: 8px; color: #1b512d;">
                    <span class="bi bi-bell-fill"></span> Notifications Récentes
                </div>
                <ul style="margin: 0; padding-left: 20px; color: #1b512d;">
                    <?php foreach ($cashierNotifications as $note): ?>
                        <li style="margin-bottom: 4px;">
                            <?php if (!empty($note['link'])): ?>
                                <a href="<?php echo htmlspecialchars($note['link'], ENT_QUOTES, 'UTF-8'); ?>" style="color: #1b512d; text-decoration: underline;">
                                    <?php echo htmlspecialchars($note['message'], ENT_QUOTES, 'UTF-8'); ?>
                                </a>
                            <?php else: ?>
                                <?php echo htmlspecialchars($note['message'], ENT_QUOTES, 'UTF-8'); ?>
                            <?php endif; ?>
                            <span style="display: block; font-size: 12px; opacity: 0.7;">
                                <?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($note['created_at'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?>
                            </span>
                        </li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <!-- Shift Status - Prominent Section -->
        <div style="background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); border-radius: 12px; padding: 25px; margin: 20px 0; box-shadow: 0 4px 15px rgba(75, 47, 31, 0.3);">
            <h3 style="color: #F4F0E4; margin: 0 0 20px 0; font-size: 24px; font-weight: 300; text-align: center;">Statut de la Session</h3>
            <?php if ($active_shift): ?>
                <div style="background-color: #FFFFFF; border-radius: 8px; padding: 20px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                        <div style="text-align: center;">
                            <div style="font-size: 12px; color: #4B2F1F; margin-bottom: 5px;">ACTIF DEPUIS</div>
                            <div style="font-size: 16px; font-weight: 600; color: #4B2F1F;"><?php echo htmlspecialchars(date('H:i', strtotime($active_shift['shift_start'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></div>
                            <div style="font-size: 12px; color: #4B2F1F;"><?php echo htmlspecialchars(date('M d, Y', strtotime($active_shift['shift_start'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></div>
                        </div>
                        <div style="text-align: center;">
                            <div style="font-size: 12px; color: #4B2F1F; margin-bottom: 5px;">DÉPOSÉE À LA BANQUE</div>
                            <div style="font-size: 18px; font-weight: 600; color: #4B2F1F;"><?php echo number_format($active_shift_bank_deposits, 0, ',', '.'); ?> BIF</div>
                        </div>
                        <div style="text-align: center;">
                            <div style="font-size: 12px; color: #4B2F1F; margin-bottom: 5px;">VENTES DE LA SESSION</div>
                            <div style="font-size: 18px; font-weight: 600; color: #F4A261;"><?php echo number_format($shift_deposits, 0, ',', '.'); ?> BIF</div>
                        </div>
                        <div style="text-align: center;">
                            <div style="font-size: 12px; color: #4B2F1F; margin-bottom: 5px;">DÉPENSES</div>
                            <div style="font-size: 18px; font-weight: 600; color: #4B2F1F;"><?php echo number_format($shift_expenses, 0, ',', '.'); ?> BIF</div>
                        </div>
                        <div style="text-align: center;">
                            <div style="font-size: 12px; color: #4B2F1F; margin-bottom: 5px;">TRANSFERTS</div>
                            <div style="font-size: 18px; font-weight: 600; color: #4B2F1F;"><?php echo number_format($shift_transfers, 0, ',', '.'); ?> BIF</div>
                        </div>
                        <div style="text-align: center; grid-column: span 2;">
                            <div style="font-size: 12px; color: #4B2F1F; margin-bottom: 5px;">SOLDE ACTUEL</div>
                            <div style="font-size: 24px; font-weight: 700; color: #4B2F1F; border: 2px solid #F4A261; border-radius: 6px; padding: 10px; background: linear-gradient(135deg, #FFF8F0 0%, #FFFFFF 100%);"><?php echo number_format($current_balance, 0, ',', '.'); ?> BIF</div>
                        </div>
                    </div>
                </div>
            <?php else: ?>
                <div style="background-color: #FFFFFF; border-radius: 8px; padding: 30px; text-align: center; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                    <div style="font-size: 48px; margin-bottom: 15px;">⚠️</div>
                    <div style="font-size: 18px; color: #4B2F1F; font-weight: 600; margin-bottom: 10px;">Aucune Session Active</div>
                    <div style="color: #666;">Veuillez démarrer une session dans Gestion des Soldes pour commencer les opérations</div>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- Available Cash - Simplified -->
        <div style="background: linear-gradient(135deg, #2A9D8F 0%, #4CAF50 100%); border-radius: 12px; padding: 25px; margin: 30px 0; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
            <h3 style="color: #FFFFFF; margin: 0 0 20px 0; font-size: 24px; font-weight: 600; text-align: center;">Argent Disponible</h3>
            <div style="background-color: rgba(255, 255, 255, 0.95); border-radius: 8px; padding: 20px; text-align: center;">
                <div style="font-size: 16px; color: #4B2F1F; margin-bottom: 15px; line-height: 1.8;">
                    <?php if ($available_from_closed_shifts > 0): ?>
                        <span style="font-weight: 600;"><?php echo number_format($available_from_closed_shifts, 0, ',', '.'); ?> BIF</span>
                        <span style="color: #666;">(Sessions fermées)</span>
                    <?php endif; ?>
                    <?php if ($active_shift && $shift_start): ?>
                        <?php if ($available_from_closed_shifts > 0): ?>
                            <span style="margin: 0 10px; color: #4B2F1F;">+</span>
                        <?php endif; ?>
                        <span style="font-weight: 600;"><?php echo number_format($current_balance, 0, ',', '.'); ?> BIF</span>
                        <span style="color: #666;">(Session active)</span>
                    <?php endif; ?>
                    <?php if ($available_from_closed_shifts > 0 || ($active_shift && $shift_start)): ?>
                        <span style="margin: 0 10px; color: #4B2F1F;">=</span>
                        <span style="font-size: 28px; font-weight: 700; color: #2A9D8F;"><?php echo number_format($argent_disponible, 0, ',', '.'); ?> BIF</span>
                    <?php else: ?>
                        <span style="font-size: 28px; font-weight: 700; color: #2A9D8F;">0 BIF</span>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Today's Metrics -->
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin: 30px 0;">
            <div style="background: linear-gradient(135deg, #FFF8F0 0%, #FFFFFF 100%); border-radius: 10px; padding: 20px; text-align: center; border-top: 4px solid #F4A261; box-shadow: 0 2px 8px rgba(0,0,0,0.1); transition: transform 0.2s;" onmouseover="this.style.transform='translateY(-3px)';" onmouseout="this.style.transform='translateY(0)';">
                <div style="font-size: 13px; color: #666; margin-bottom: 8px; text-transform: uppercase; letter-spacing: 0.5px;">VENTES AUJOURD'HUI</div>
                <div style="font-size: 28px; font-weight: 700; color: #F4A261;"><?php echo number_format($total_sales ?? 0, 0, ',', '.'); ?> BIF</div>
            </div>
            <div style="background: linear-gradient(135deg, #F5F5F5 0%, #FFFFFF 100%); border-radius: 10px; padding: 20px; text-align: center; border-top: 4px solid #4B2F1F; box-shadow: 0 2px 8px rgba(0,0,0,0.1); transition: transform 0.2s;" onmouseover="this.style.transform='translateY(-3px)';" onmouseout="this.style.transform='translateY(0)';">
                <div style="font-size: 13px; color: #666; margin-bottom: 8px; text-transform: uppercase; letter-spacing: 0.5px;">COMMANDES EN ATTENTE</div>
                <div style="font-size: 28px; font-weight: 700; color: #4B2F1F;"><?php echo number_format($pending_orders ?? 0, 0, ',', '.'); ?></div>
            </div>
            <div style="background: linear-gradient(135deg, #E8F5E9 0%, #FFFFFF 100%); border-radius: 10px; padding: 20px; text-align: center; border-top: 4px solid #2E7D32; box-shadow: 0 2px 8px rgba(0,0,0,0.1); transition: transform 0.2s;" onmouseover="this.style.transform='translateY(-3px)';" onmouseout="this.style.transform='translateY(0)';">
                <div style="font-size: 13px; color: #666; margin-bottom: 8px; text-transform: uppercase; letter-spacing: 0.5px;">VENTES À CRÉDIT (AUJOURD'HUI)</div>
                <div style="font-size: 28px; font-weight: 700; color: #2E7D32;"><?php echo number_format($credit_sales_today ?? 0, 0, ',', '.'); ?> BIF</div>
                <div style="font-size: 12px; color: #666; margin-top: 5px;">(<?php echo number_format($credit_sales_count ?? 0, 0, ',', '.'); ?> ventes)</div>
            </div>
            <?php if ($active_shift): ?>
            <div style="background: linear-gradient(135deg, #E3F2FD 0%, #FFFFFF 100%); border-radius: 10px; padding: 20px; text-align: center; border-top: 4px solid #1976D2; box-shadow: 0 2px 8px rgba(0,0,0,0.1); transition: transform 0.2s;" onmouseover="this.style.transform='translateY(-3px)';" onmouseout="this.style.transform='translateY(0)';">
                <div style="font-size: 13px; color: #666; margin-bottom: 8px; text-transform: uppercase; letter-spacing: 0.5px;">ARGENT ACCUMULÉ (SESSION)</div>
                <div style="font-size: 28px; font-weight: 700; color: #1976D2;"><?php echo number_format($accumulated_cash_this_shift, 0, ',', '.'); ?> BIF</div>
                <div style="font-size: 12px; color: #666; margin-top: 5px;">Depuis le début de la session</div>
            </div>
            <?php endif; ?>
        </div>
        <div style="background-color: #FFFFFF; border-radius: 12px; padding: 25px; margin: 30px 0; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
            <h3 style="color: #4B2F1F; margin: 0 0 20px 0; font-size: 24px; font-weight: 600; text-align: center; padding-bottom: 15px; border-bottom: 2px solid #E0E0E0;">Statut des Commandes du Dépôt</h3>
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px;">
            <div style="background: linear-gradient(135deg, #F4A261 0%, #d88a4f 100%); padding: 15px; border-radius: 10px; box-shadow: 0 3px 12px rgba(244, 162, 97, 0.3); text-align: center; transition: transform 0.3s ease, box-shadow 0.3s ease; cursor: pointer;" onmouseover="this.style.transform='translateY(-3px)'; this.style.boxShadow='0 6px 20px rgba(244, 162, 97, 0.4)';" onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 3px 12px rgba(244, 162, 97, 0.3)';">
                <div style="font-size: 24px; margin-bottom: 5px; color: #F4F0E4;"><i class="bi bi-clock-history"></i></div>
                <h4 style="margin: 0; color: #F4F0E4; font-size: 12px; font-weight: 500;">En Attente</h4>
                <p style="margin: 5px 0 0; color: #F4F0E4; font-size: 20px; font-weight: 700;"><?php echo number_format($province_invoice_summary['pending'], 0, ',', '.'); ?></p>
            </div>
            <div style="background: linear-gradient(135deg, #28a745 0%, #20c997 100%); padding: 15px; border-radius: 10px; box-shadow: 0 3px 12px rgba(40, 167, 69, 0.3); text-align: center; transition: transform 0.3s ease, box-shadow 0.3s ease; cursor: pointer;" onmouseover="this.style.transform='translateY(-3px)'; this.style.boxShadow='0 6px 20px rgba(40, 167, 69, 0.4)';" onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 3px 12px rgba(40, 167, 69, 0.3)';">
                <div style="font-size: 24px; margin-bottom: 5px; color: #F4F0E4;"><i class="bi bi-check-circle-fill"></i></div>
                <h4 style="margin: 0; color: #F4F0E4; font-size: 12px; font-weight: 500;">Approuvé</h4>
                <p style="margin: 5px 0 0; color: #F4F0E4; font-size: 20px; font-weight: 700;"><?php echo number_format($province_invoice_summary['approved'], 0, ',', '.'); ?></p>
            </div>
            <div style="background: linear-gradient(135deg, #dc3545 0%, #c82333 100%); padding: 15px; border-radius: 10px; box-shadow: 0 3px 12px rgba(220, 53, 69, 0.3); text-align: center; transition: transform 0.3s ease, box-shadow 0.3s ease; cursor: pointer;" onmouseover="this.style.transform='translateY(-3px)'; this.style.boxShadow='0 6px 20px rgba(220, 53, 69, 0.4)';" onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 3px 12px rgba(220, 53, 69, 0.3)';">
                <div style="font-size: 24px; margin-bottom: 5px; color: #F4F0E4;"><i class="bi bi-x-circle-fill"></i></div>
                <h4 style="margin: 0; color: #F4F0E4; font-size: 12px; font-weight: 500;">Rejeté</h4>
                <p style="margin: 5px 0 0; color: #F4F0E4; font-size: 20px; font-weight: 700;"><?php echo number_format($province_invoice_summary['rejected'], 0, ',', '.'); ?></p>
            </div>
            <div style="background: linear-gradient(135deg, #007bff 0%, #0056b3 100%); padding: 15px; border-radius: 10px; box-shadow: 0 3px 12px rgba(0, 123, 255, 0.3); text-align: center; transition: transform 0.3s ease, box-shadow 0.3s ease; cursor: pointer;" onmouseover="this.style.transform='translateY(-3px)'; this.style.boxShadow='0 6px 20px rgba(0, 123, 255, 0.4)';" onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 3px 12px rgba(0, 123, 255, 0.3)';">
                <div style="font-size: 24px; margin-bottom: 5px; color: #F4F0E4;"><i class="bi bi-cash-coin"></i></div>
                <h4 style="margin: 0; color: #F4F0E4; font-size: 12px; font-weight: 500;">Payé</h4>
                <p style="margin: 5px 0 0; color: #F4F0E4; font-size: 20px; font-weight: 700;"><?php echo number_format($province_invoice_summary['paid'], 0, ',', '.'); ?></p>
            </div>
            <div style="background: linear-gradient(135deg, #6f42c1 0%, #5a32a3 100%); padding: 15px; border-radius: 10px; box-shadow: 0 3px 12px rgba(111, 66, 193, 0.3); text-align: center; transition: transform 0.3s ease, box-shadow 0.3s ease; cursor: pointer;" onmouseover="this.style.transform='translateY(-3px)'; this.style.boxShadow='0 6px 20px rgba(111, 66, 193, 0.4)';" onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='0 3px 12px rgba(111, 66, 193, 0.3)';">
                <div style="font-size: 24px; margin-bottom: 5px; color: #F4F0E4;"><i class="bi bi-check2-all"></i></div>
                <h4 style="margin: 0; color: #F4F0E4; font-size: 12px; font-weight: 500;">Servi</h4>
                <p style="margin: 5px 0 0; color: #F4F0E4; font-size: 20px; font-weight: 700;"><?php echo number_format($province_invoice_summary['served'], 0, ',', '.'); ?></p>
            </div>
            </div>
        </div>
    </div>
</div>
<?php require_once __DIR__ . '/../includes/footer.php'; ?>
